------------------------------------------------------------------------------
-- Render Manager Status, Revision: 2.0 
--
-- comp script
--
-- This simple script connects to the render master configured in the 
-- preferences and outputs a list of slaves, and then prints a list of 
-- jobs in the queue to the console.
--
-- Written as an example of a scripting the render manager, the example is 
-- still useful for monitoring your jobs on a remote render manager from 
-- within Fusion. It should be trivial to modify this script to run from 
-- the command line, or even produce output for a webpage.
--
-- written by : Isaac Guenard (izyk@eyeonline.com)
-- written    : August 7th, 2003
-- updated : Sept 27, 2005
-- changes : updated for 5
------------------------------------------------------------------------------

function getfilename(path)
	for i = string.len(path), 1, -1 do
		teststring = string.sub(path, i, i)
		if teststring == "\\" or teststring == "/" then
			return string.sub(path, i+1)
		end
	end
end

function fmtSlave(slave)
	slv = slave:GetAttrs()

	if slv.RSLVS_PriorityClasses == nil then classes = "N/A" else classes = slv.RSLVS_PriorityClasses end
	
	if slv.RSLVN_Status == 5 then
		status = getfilename(slave:GetJob():GetAttrs().RJOBS_Name)
	else
		status = slv.RSLVS_Status
	end
	
	return string.format("%-40.40s", slv.RSLVS_Name).." "..
	string.format("%-16.16s", slv.RSLVS_IP  ).." "..
	string.format("%-15.15s", slv.RSLVS_Version).." "..
	string.format("%-30.30s", status).." "..
	string.format("%-10.10s", classes)
end

function fmtJob(job)
	a = job:GetAttrs()

    total = a.RJOBN_UnrenderedFrames + a.RJOBN_RenderedFrames + a.RJOBN_RenderingFrames
    done  = a.RJOBN_RenderedFrames

	if a.RJOBS_PriorityClass == nil then
	   classes = nil
	else
	   classes = a.RJOBS_PriorityClass
	end
  
	if a.RJOBS_Type == "Comp" then
		
		local strAssembly = string.format("%-40.40s", bmd.parseFilename(a.RJOBS_Name).FullName).." "..
		string.format("%-16.16s", a.RJOBS_QueuedBy).." "..
		string.format("%-15.15s", done.."/"..total).." "..
		string.format("%-30.30s", a.RJOBS_Status).." "
		if classes then
			strAssembly=strAssembly..string.format("%-10.10s", classes)
		end
		return strAssembly
	end
end

-----------------------------------------------------------------------------------------------
--- Main Body
---
-----------------------------------------------------------------------------------------------
if fusion == nil then
	fusion = Fusion()
	if fusion == nil then 
		print("Could not connect to a Digital Fusion")
		return	
	end
end

-- where is the master
master = fusion:GetPrefs()["Global"]["Network"]["ServerName"]

if master == "" then master = "localhost" end

remote = Fusion(master, 10, nil, "RenderManager")
if remote == nil then 
	print("Could not connect to the Render Master")
	return
end

rm = remote.RenderManager
if rm == nil then
	print("Could not connect to render manager!")
	exit()
end

-- slave and job lists
jobs   = rm:GetJobList()
slaves = rm:GetSlaveList()

print()
print("-----  status of render manager : "..master.."  "..string.rep("-", 85 - string.len(master) ) )
print()
print("[ slave list ]                           [ ip address ]   [ version ]     [ current status ]             [ classes ]     ")
print("--------------                           --------------   -----------     ------------------             -----------     ")

-----------------------------------------------------------------------------------------------
--- The slave list
---
-----------------------------------------------------------------------------------------------
if table.getn(slaves) == 0 then 
  print("No Slaves In Queue")
end

for i, v in pairs(slaves) do
  print( fmtSlave(v, i) )
end

print()
print()
print("[ comps in queue ]                       [ queued by ]    [ frames ]      [ current status ]             [ classes ]     ")
print("------------------                       -------------    ----------      ------------------             -----------     ")
-----------------------------------------------------------------------------------------------
--- The job list
---
-----------------------------------------------------------------------------------------------	

if table.getn(jobs) == 0 then 
	print("No Jobs In Queue")
end

for i, v in pairs(jobs) do
	print( fmtJob(v))
end

print()
print("-------------------------------------------------------------------------------------------------------------  done  ----")


